/**
* \file: AlsaAudioSinkImpl.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
*
* \copyright (c) 2014-2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef ALSAAUDIOSINKIMPL_H
#define ALSAAUDIOSINKIMPL_H

#include <aauto/AudioSink.h>
#include <aauto/GalReceiver.h>
#include <aauto/util/scoped_ptr.h>
#include <aauto/util/shared_ptr.h>
#include <aauto/util/WorkQueue.h>
#include <aauto/util/IoBuffer.h>

#include "aauto/AditAudioSink.h"
#include <uspi/ConfigHandle.h>
#include "AlsaAudioCommon.h"
#include "AlsaAudioSinkRecord.h"

#define MAX_AUDIO_UNACKED_FRAMES            4096

#define MEDIA_AUDIO_SAMPLING_RATE           48000
#define MEDIA_AUDIO_BITS_PER_SAMPLE         16
#define MEDIA_AUDIO_CHANNELS                2

#define SYSTEM_AUDIO_SAMPLING_RATE          16000
#define SYSTEM_AUDIO_BITS_PER_SAMPLE        16
#define SYSTEM_AUDIO_CHANNELS               1

#define GUIDANCE_AUDIO_SAMPLING_RATE        16000
#define GUIDANCE_AUDIO_BITS_PER_SAMPLE      16
#define GUIDANCE_AUDIO_CHANNELS             1

namespace adit { namespace aauto
{
/* based on enum AudioStreamType */
static const char *audioSinkName[] =
{
        "AudioSinkUnknown",
        "AudioSinkGuidance",    // AudioStreamType shall be AUDIO_STREAM_GUIDANCE       1
        "AudioSinkSytemAudio",  // AudioStreamType shall be AUDIO_STREAM_SYSTEM_AUDIO   2
        "AudioSinkMedia",       // AudioStreamType shall be AUDIO_STREAM_MEDIA          3
        "AudioSinkTelephony"    // AudioStreamType shall be AUDIO_STREAM_TELEPHONY      4
};

class AlsaSinkConfig : public uspi::ConfigHandle
{
public:
AlsaSinkConfig()
{
	mConfSType = "";
    mConfCodec = "";
    mSampleRate = 0;
    mInitToutms = 0;
    mNumBits = 0;
    mNumChannels = 0;
    mUnackedFrames = 0;
    mVerbose = 0;
    mDiscardFrames = 0;
    mAlsaLogging = 0;
    mDisablePrio = 0;
    mThreadPrio = 0;
    mPeriodms = 0;
    mBufferperiods = 0;
    mSilencems = 0;
    mDevice = "";
    mStreamType = 0;
    mCodec = 0;
    mThresholdMs = 0;
    mAudioRecord = 0;
    mAudioRecordFile = "";
}

bool ResultConfig()
{
    bool isValid = true;
    mConfSType      = getString("audio-sink-stream-type", &isValid, Range(-1, INT_MAX));
    mConfCodec      = getString("audio-sink-stream-codec", &isValid, Range(-1, INT_MAX));
    mSampleRate     = getInt("audio-sink-sampling-rate", &isValid, Range(-1, INT_MAX), MEDIA_AUDIO_SAMPLING_RATE);
    mNumBits        = getInt("audio-sink-bits-per-sample", &isValid, Range(-1, INT_MAX), MEDIA_AUDIO_BITS_PER_SAMPLE);
    mNumChannels    = getInt("audio-sink-channels", &isValid, Range(-1, INT_MAX), MEDIA_AUDIO_CHANNELS);
    mUnackedFrames  = getInt("audio-sink-max-unacked-frames", &isValid, Range(-1, INT_MAX), MAX_AUDIO_UNACKED_FRAMES);
    mDiscardFrames  = getInt("audio-sink-discard-frames", &isValid, Range(0, 1), 0);
    mVerbose        = getInt("enable-verbose-logging", &isValid, Range(0, 1), 0);
    mAlsaLogging    = getInt("alsa-dump-configuration", &isValid, Range(0, 1), 0);
    mDisablePrio    = getInt("disable-real-time-priority-audio", &isValid, Range(0, 1), 0);
    mThreadPrio     = getInt("audio-threads-real-time-priority", &isValid, Range(-1, INT_MAX), 61);
    mThresholdMs    = getInt("audio-sink-threshold-ms", &isValid, Range(0, INT_MAX), 0);
    mAudioRecord    = getIntMultiKey("audio-sink-record", "enable", &isValid, Range(-1, 1), 0);
    mAudioRecordFile = getStringMultiKey("audio-sink-record", "filename", &isValid, Range(-1, INT_MAX), "");
    mPeriodms       = getIntMultiKey("alsa-main-audio", "periodms", &isValid, Range(-1, INT_MAX));
    mBufferperiods  = getIntMultiKey("alsa-main-audio", "bufferperiods", &isValid, Range(-1, INT_MAX), -1);
    mSilencems      = getIntMultiKey("alsa-main-audio", "silencems", &isValid, Range(-1, INT_MAX), -1);
    mInitToutms     = getIntMultiKey("alsa-main-audio", "inittoutms", &isValid, Range(-1, INT_MAX), 500);
    mDevice         = getStringMultiKey("alsa-main-audio", "device", &isValid, Range(-1, INT_MAX));

    return isValid;
}
    string  mConfSType;     // Streaming type
    string  mConfCodec;     // Codec type
    int     mSampleRate;    // Sampling rate
    int     mNumBits;       // Bits per sample
    int     mNumChannels;   // Number of Channels
    int     mUnackedFrames; // unacknowledged Frames

    /* if set true, work queue will be flushed */
    bool    mDiscardFrames;
    bool    mVerbose;       // Verbose log flag 
    bool    mAlsaLogging;   // Alsa log flag 
    bool    mDisablePrio;   // Priority Set flag
    int     mThreadPrio;    // Thread Priority
    string  mAlsaSinkSetting;   // Alsa Common settings data

    int     mStreamType;    // Streaming type
    int     mCodec;         // Codec type

    int     mThresholdMs;

    int     mAudioRecord;   // Enable record of received audio data
    string  mAudioRecordFile;

    // ALSA device setting
    int     mInitToutms;
    int     mPeriodms;
    int     mBufferperiods;
    int     mSilencems;
    string  mDevice;
};

class AlsaAudioSinkImpl : public IAudioSinkCallbacks
{
public:
    AlsaAudioSinkImpl(AudioSink* inSink);
    ~AlsaAudioSinkImpl();
    bool init();
    void shutdown();

    void setConfigItem(string inKey, string inValue);
    void registerCallbacks(IAditAudioSinkCallbacks* inCallbacks);

    const char* sinkName(int AudioStreamType) { return audioSinkName[AudioStreamType]; };
protected:
    /* IAudioSinkCallbacks */
    void dataAvailableCallback(int32_t sessionId, uint64_t timestamp,
            uint8_t* data, size_t len);
    void dataAvailableCallback(int32_t sessionId, uint64_t timestamp,
            const shared_ptr<IoBuffer>& galFrame, uint8_t* data, size_t len);
    int codecConfigCallback(uint8_t* data, size_t len);
    int setupCallback(int mediaCodecType);
    void playbackStartCallback(int32_t sessionId);
    void playbackStopCallback(int32_t sessionId);

private:
    class AudioPlaybackWorkItem : public WorkItem
    {
    public:
        AudioPlaybackWorkItem(AlsaAudioSinkImpl* sink, int32_t sessionId,
                uint64_t timestamp, const shared_ptr<IoBuffer>& data,
                uint8_t* dataPtr, size_t len)
                : mSink(sink), mSessionId(sessionId), mTimestamp(timestamp),
                  mData(data), mDataPtr(dataPtr), mLen(len) { }
    protected:
        // WorkItem implementation
        void run();
    private:
        void SetThreadParam(const char* threadName);

        AlsaAudioSinkImpl* mSink;
        int32_t mSessionId;
        uint64_t mTimestamp;
        shared_ptr<IoBuffer> mData;
        uint8_t* mDataPtr;
        size_t mLen;
    };

    /**
     * @brief  Helper API to get the current configuration
     *         and set the internal members (e.g.mSampleRate, etc.)
     */
    bool getCurrConfig(void);

    bool mInitialized;
    bool mStarted;
    int32_t mSessionId;
    bool mQueueRunning;
    string mSinkName;
    bool mShutdown;

    shared_ptr<WorkQueue> mWorkQueue;

    IAditAudioSinkCallbacks* mCallbacks;
    AudioSink* audioSink;
    scoped_ptr<AlsaAudioCommon> mPlayer;
    bool mIsSetThreadParam;

    /**
     * @brief Helper API, called in case of playbackStartCallback from MD.
     *        Prepares ALSA device, calculates pre-buffering and creates work queue.
     */
    bool start();
    /**
     * @brief Helper API, called in case of playbackStopCallback from MD or shutdown triggered by Application.
     *        Stop and release ALSA device and shuts down the work queue.
     *
     * @param discardRemaining - if true, discards all remaining work items
     */
    void stop(bool discardRemaining);
    /**
     * @brief Helper API, called by helper function start().
     *        Set up the ALSA device by using the configuration set by Application.
     */
    bool prepare();

    AlsaSinkConfig  mConfig;

    AlsaAudioSinkRecord mAudioSinkRecord;
    bool AudioSinkRecordRunning;
};
} } /* namespace adit { namespace aauto */

#endif // ALSAAUDIOSINKIMPL_H
